/** @file   ammunition.h
 * @brief   Declaration of Ammunition - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:41 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_AMMUNITION_H
#define H_WWW_AMMUNITION_H

#include "eng2d.h"

namespace WeWantWar {

// Let the compiler know about these classes.
class GameObject;
class BulletTable;

/** @class  Ammunition
 * @brief   This class operates as a baseclass for Bullet and Grenade - classes.
 * @author  Tomi Lamminsaari
 *
 * This baseclass provides basic functionality so that the ammunition can move
 * and animate.
 *
 * The update()-method alters the altitude but it does not react when it
 * touches the ground level. This is done by decreasing the value of
 * @c iVerticalSpeed by amount of Consts::GRAVITY. Then we add this speed
 * value to the @c iAltitude - member that tells the altitude.
 *
 * You must implement the @c updateAmmunition() - method in the child class.
 * The main @c update() - method calls that method so that you can write new
 * behaviour to your ammunition object.
 */
class Ammunition
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** A datatype for bitfield flags. */
  typedef unsigned int TFlags;

  /** A bitmask for bounce-flag. When this flag is set, this Ammunition bounces
   * from the collidable map tiles. If this flag is not set, this Ammunition
   * gets deleted as it collides with map tiles.
   */
  static const TFlags KBounce = 1;
  /** A bitmask for "leave trails"-flag. When this flag is set, this Ammunition
   * leaves a visual effect as it moves.
   */
  static const TFlags KTrails = 1 << 1;
  /** A bitmask for "make sound"-flag. When set, this Ammunition makes sound
   * as it touches to the collidable map tiles.
   */
  static const TFlags KMakeSound = 1 << 2;
  

  ///
  /// Public members
  /// ==============
  
  /** Pointer to GameObject who spawned this Ammunition. */
  GameObject* iOwner;
  
  /** 2d-position of this object in pixes from topleft corner of the map. */
  eng2d::Vec2D  iPosition;
  
  /** Altitude of this object */
  float iAltitude;
  
  /** The vertical speed of this object. Gets decreased by Consts::GRAVITY in
   * every update - round.
   */
  float iVerticalSpeed;
  
  /** The bitfield flags this Ammunition has such as @c KBounce */
  TFlags iFlags;

  /** The animation for this Ammunition. */
  eng2d::Animation  iAnimation;
  
  /** A flag that indicates when the manager-object should delete us. */
  bool iDone;
  
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Ammunition();

  /** Constructs new Ammunition that was created by the given GameObject.
   * @param     aOwner            Pointer to GameObject who created this
   */
  Ammunition( GameObject* aOwner );

	/** Destructor
   */
	virtual ~Ammunition();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Ammunition
   */
  Ammunition( const Ammunition& rO );

	/** Assignment operator
   * @param     rO                Reference to another Ammunition
   * @return    Reference to us.
   */
  Ammunition& operator = ( const Ammunition& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this object. This method moves us based on the speed settings.
   * This also checks the collisions to the tilemap.
   * @return    <code>true</code> if this object should be deleted.
   */
  bool update();
  
  /** This method should update the actual ammunition. If your ammunition
   * needs any special behaviour, you should place it in this method. This
   * method gets called from the @c update() - method.
   *
   * To make the manager object ( BulletTable - class ) destroy this object,
   * you should return a @c true - value.
   * @return    <code>true</code> if we've done with this object and it should
   *            be deleted.
   */
   virtual bool updateAmmunition() = 0;

   
  /** Redraws this Ammunition.
   * @param     aTarget           The target bitmap.
   */
  virtual void redraw( BITMAP* aTarget ) const = 0;
  
  /** This method is called when this ammunition's lifecircle has ended. If
   * you want to make ammunition that explode once they hit something or they've
   * reach their maximum range/lifetime, you can write the ending actions here.
   *
   * Default implementation does nothing so it's sufficient for most of the
   * regular bullets.
   * @param   aBulletTable      Pointer to BulletTable in case there's need to
   *                            spawn new bullets.
   */
  virtual void endingActions( BulletTable* aBulletTable );
  
  /** Sets the velocity
   * @param     aVelocity         The velocity-vector
   */
  virtual void setVelocity( const eng2d::Vec2D& aVelocity );
  
  

  ///
  /// Getter methods
  /// ==============
  
  /** Tells if requested flag is on.
   * @param     aFlagMask         The bitmask that should have only one
   *                              bit on. If there are more than one bit one,
   *                              this returns true if at least one of them
   *                              is on.
   * @return    true if at least one of the given flags are on.
   */
  virtual bool hasFlag( TFlags aFlagMask ) const;

  /** Returns the velocity.
   * @return    The velocity vector.
   */
  inline eng2d::Vec2D velocity() const
  {
    return iVelocity;
  }
  
  

protected:

  /** Makes this ammunition to bounce from collidable map tiles.
   */
  virtual void bounce();
  

  ///
  /// Members
  /// =======

  /** The velocity of this object. */
  eng2d::Vec2D  iVelocity;
  
  /** Length of the velocity vector. */
  float iVelocityVectorLength;


private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: ammunition.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:36:46+02  lamminsa
 * endingActions() method was added.
 *
 */
 
